//=============================================================================
// Entspricht Telefonbuch 4_2, nur zusaetzlich Strasse und Ort im Eintrag.

#include <algorithm>
#include <cctype>
#include <fstream>
#include <iostream>
#include <map>
#include <string>
#include <vector>
using namespace std;

//-----------------------------------------------------------------------------

const string filename("telefonbuch4.3.txt");

//-----------------------------------------------------------------------------

bool starts_with(const string& src, const string& pattern)
{
   if (src.length() < pattern.length())
   {
      return false;
   }
   return src.substr(0, pattern.length()) == pattern;
}

//-----------------------------------------------------------------------------
// Achtung: "std::tolower" funktioniert nur mit den amerikanischen Zeichen "A-Z",
// und nicht mit zum Beispiel Umlauten.

string to_lower(const string& s)
{
   string res(s.length(), ' ');
   transform(cbegin(s), cend(s), begin(res), [](unsigned char c) { return static_cast<char>(tolower(c)); });
   return res;
}

//-----------------------------------------------------------------------------

class entry
{
public:
   string const& name() const { return name_; }

   void input(string const& name);
   void print() const;

   void load(ifstream& in, int& line);
   void write(ofstream& out) const;

private:
   using numbers = vector<string>;
   string name_;
   string street_;
   string town_;
   numbers numbers_;
};

//-----------------------------------------------------------------------------

class book
{
public:
   void new_entry();

   void print() const;
   void find_and_print() const;

   void load(const string& filenname);
   bool write(const string& filenname) const;

private:
   using entries = map<string, entry>;
   entries entries_;
};

//-----------------------------------------------------------------------------

void entry::input(string const& name)
{
   name_ = name;

   cout << "> Strasse: ";
   getline(cin, street_);

   cout << "> Ort: ";
   getline(cin, town_);

   for (string no;;)
   {
      cout << "> Nr.: ";
      getline(cin, no);
      if (no.empty()) break;
      numbers_.push_back(no);
   }
}

//-----------------------------------------------------------------------------

void entry::print() const
{
   cout << "* " << name_
      << ", " << street_
      << ", " << town_
      << " => ";
   for (const string& no : numbers_)
   {
      cout << '(' << no << ") ";
   }
   cout << '\n';
}

//-----------------------------------------------------------------------------

void entry::load(ifstream& in, int& line)
{
   string name, street, town;

   ++line;
   getline(in, name);
   if (in.fail()) return;

   ++line;
   getline(in, street);
   if (in.fail()) return;

   ++line;
   getline(in, town);
   if (in.fail()) return;

   name_ = name;
   street_ = street;
   town_ = town;

   for (string no;;)
   {
      ++line;
      getline(in, no);
      // Bei s.empty() Schleife beenden, da eine Leerzeile 
      // das Ende der Telefon-Nummern anzeigt.
      if (in.fail() || no.empty()) break;
      numbers_.push_back(no);
   }
}

//-----------------------------------------------------------------------------

void entry::write(ofstream& out) const
{
   out << name_ << '\n'
      << street_ << '\n'
      << town_ << '\n';

   for (const string& no : numbers_)
   {
      out << no << '\n';
   }
   out << '\n';
}

//-----------------------------------------------------------------------------

void book::new_entry()
{
   cout << "Neuer Eintrag\n";

   string name;
   cout << "> Name: ";
   getline(cin, name);

   if (name.empty())
   {
      cout << "Abbruch 'neuer Eintrag', da Name leer\n";
      return;
   }

   string lower_name(to_lower(name));
   if (entries_.count(lower_name) > 0)
   {
      cout << "Abbruch 'neuer Eintrag', da Name schon vorhanden\n";
      return;
   }

   entry& e = entries_[lower_name];
   e.input(name);
}

//-----------------------------------------------------------------------------

void book::print() const
{
   cout << entries_.size() << " Eintraege:\n";
   for (auto& [name, numbers] : entries_)
   {
      numbers.print();
   }
}

//-----------------------------------------------------------------------------

void book::find_and_print() const
{
   string pattern;
   cout << "Suchen nach: ";
   getline(cin, pattern);

   string lower_pattern(to_lower(pattern));

   auto lower = entries_.lower_bound(lower_pattern);
   auto eit = cend(entries_);

   if (lower == eit || !starts_with(lower->first, lower_pattern))
   {
      cout << "\nKein Name im Telefonbuch beginnt mit \"" << pattern << "\".\n";
      return;
   }

   cout << "\nGefunden:\n";
   for (; lower != eit; ++lower)
   {
      if (!starts_with(lower->first, lower_pattern)) break;
      lower->second.print();
   }
}

//-----------------------------------------------------------------------------

void book::load(const string& filename)
{
   cout << "Lade Daten aus der Datei \"" << filename << "\"...\n";

   ifstream in(filename);
   if (in.fail())
   {
      cout << "- Probleme beim Oeffnen der Datei.\n- 0 Eintraege geladen.\n";
      return;
   }

   int line = 0;
   do
   {
      entry e;
      e.load(in, line);
      if (in.fail()) break;
      string lower_name(to_lower(e.name()));
      entries_.insert({ lower_name, e });
   } while (!in.fail());
   if (!in.eof())
   {
      cout << "- Problem beim Lesen der Zeile " << line << '\n';
   }

   cout << "- " << entries_.size() << " Eintrage geladen.\n";
}

//-----------------------------------------------------------------------------

bool book::write(const string& filename) const
{
   cout << "Sichere Daten in die Datei \"" << filename << "\"...\n";

   ofstream out(filename);
   if (out.fail())
   {
      cout << "- Probleme beim Oeffnen der Datei.\n";
      return false;
   }

   for (auto& [name, numbers] : entries_)
   {
      numbers.write(out);
      if (out.fail())
      {
         cout << "- Probleme beim Schreiben in die Datei.\n";
         return false;
      }
   }

   cout << "- fertig\n";
   return true;
}

//-----------------------------------------------------------------------------

int main()
{
   cout << "Aufg_14_03_Telefonbuch_4_3\n\n";

   book b;
   b.load(filename);

   for (string s;;)
   {
      cout << "\nBitte waehlen Sie eine Aktion: \n"
         << "- e : Programmende\n"
         << "- l : Alle Eintraege auflisten\n"
         << "- s : Nach Eintrag suchen\n"
         << "- n : Neuen Eintrag eingeben\n"
         << "> ";
      getline(cin, s);
      if (s.length() != 1) continue;
      cout << '\n';

      switch (s[0])
      {
      case 'e':
         if (b.write(filename))
         {
            cout << "\nAuf Wiedersehen\n";
            return 0;
         }
         break;
      case 'l':
         b.print();
         break;
      case 's':
         b.find_and_print();
         break;
      case 'n':
         b.new_entry();
         break;
      }
   }
}
